--
-- FS22 - MajorService
--
-- @Interface: 1.0.0.0
-- @Author: Time Wasting Productions
-- @Date: 25.10.2022
-- @Version: 1.0.0.1
--
-- Changelog:
-- 	v1.0.0.0 (20.10.2022):
--      - Initial Release
-- 	v1.0.0.1 (25.10.2022):
--      - Add support Animal Pen Pallets (Eggs/Wool)
--      - Add German Translations
--      - ModHub Release

source(g_currentModDirectory.."WearableMajorServiceEvent.lua") --multiplayer event file

MajorService = {}
MajorService.dir = g_currentModDirectory
MajorService.modName = g_currentModName
MajorService.debug = false
MajorService.Cost = 0.05

function MajorService.prerequisitesPresent(specializations)
	if MajorService.debug then
		print("---- MajorService: prerequisitesPresent")
	end
    return SpecializationUtil.hasSpecialization(Wearable, specializations);
end

function MajorService.initSpecialization()

	if MajorService.debug then
		print("---- MajorService: initSpecialization")
	end

	--g_gui:loadGui(MajorService.dir .. "gui/myWorkshopScreen.xml", "myWorkshopScreen", my_workshopScreen)
	g_workshopScreen.xmlFilename = MajorService.dir .. "gui/myWorkshopScreen.xml"
	WorkshopScreen.createFromExistingGui(g_workshopScreen, "WorkshopScreen")

	local schemaSavegame = Vehicle.xmlSchemaSavegame
    schemaSavegame:register(XMLValueType.FLOAT, "vehicles.vehicle(?).FS25_MajorService.MajorService#lastServiceAt", "Last Major Service At")
    schemaSavegame:register(XMLValueType.FLOAT, "vehicles.vehicle(?).FS25_MajorService.MajorService#currentHours", "Vehicle Current Hours")

    Wearable.updateDamageAmount = Utils.overwrittenFunction(Wearable.updateDamageAmount, MajorService.updateDamageAmount)
	WorkshopScreen.populateCellForItemInSection = Utils.appendedFunction(WorkshopScreen.populateCellForItemInSection, MajorService.populateCellForItemInSection)
	WorkshopScreen.onListSelectionChanged = Utils.appendedFunction(WorkshopScreen.onListSelectionChanged, MajorService.onListSelectionChanged)
	local workshopMenu = g_workshopScreen
 

	if FS25_Configure_Maintenance ~= nil and FS25_Configure_Maintenance.ReduceMaintenance ~= nil then
		FS25_Configure_Maintenance.ReduceMaintenance.updateDamageAmount = Utils.overwrittenFunction(FS25_Configure_Maintenance.ReduceMaintenance.updateDamageAmount, MajorService.updateDamageAmount);
	end
	

	if workshopMenu.serviceButton == nil then
		if MajorService.debug then
			print("---- MajorService: create serviceButton")
		end
		local parent = workshopMenu.sellButton.parent
		workshopMenu.serviceButton = workshopMenu.sellButton:clone(parent)
		workshopMenu.serviceButton.onClickCallback = MajorService.onClickServiceCallback
		workshopMenu.serviceButton:setText("service")
		workshopMenu.serviceButton:setInputAction(InputAction.MENU_PAGE_PREV)
		workshopMenu.serviceButton:setVisible(true)
		g_workshopScreen.serviceButton.parent:invalidateLayout()
	end
end

function MajorService.populateCellForItemInSection(self, list, section, index, cell)
	local vehicle = g_workshopScreen.vehicles[index]

	if MajorService.debug then
		print("---- MajorService: MajorService:populateCellForItemInSection")
	end
	
	cell:getAttribute("lastMSText"):setText(MajorService:setLastServiceText(vehicle.MajorService.lastServiceAt))
	cell:getAttribute("nextMSText"):setText(MajorService:setNextServiceText(vehicle.MajorService.lastServiceAt, vehicle.operatingTime, vehicle.lifetime))
end


function MajorService.registerOverwrittenFunctions(vehicleType)
	if MajorService.debug then
		print("---- MajorService: registerOverwrittenFunctions")
	end
    SpecializationUtil.registerOverwrittenFunction(vehicleType, "updateDamageAmount", MajorService.updateDamageAmount);
end



function MajorService:onClickServiceCallback()
	if MajorService.debug then
		print("---- MajorService: onClickServiceCallback")
	end

	local vehicle = g_workshopScreen.vehicle
	if vehicle ~= nil and vehicle:getServiceCost() >= 1 then
		YesNoDialog.show(MajorService.onYesNoServiceDialog, 
					MajorService, 
					string.format(g_i18n:getText("ui_mvServiceDialog"), g_i18n:formatMoney(vehicle:getServiceCost())), 
					nil, 
					g_i18n:getText("button_continue"), 
					g_i18n:getText("button_cancel"), 
					DialogElement.TYPE_QUESTION, 
					GuiSoundPlayer.SOUND_SAMPLES.CONFIG_WRENCH, 
					nil, 
					{["vehicle"] = vehicle})

		return true
	else
		return false
	end

end
function MajorService:onYesNoServiceDialog(yes, args)
	local vehicle = args.vehicle
	if MajorService.debug then
		print("---- MajorService: onYesNoServiceDialog ".. vehicle:getName())
	end
	if yes then
		if g_currentMission:getMoney() < vehicle:getServiceCost() then
			InfoDialog.show(g_i18n:getText("shop_messageNotEnoughMoneyToBuy"))
		else
			g_client:getServerConnection():sendEvent(WearableMajorServiceEvent.new(vehicle))
		end
	end
end

function MajorService:serviceVehicle()
	if MajorService.debug then
		print("---- MajorService: serviceVehicle")
	end
	--if self.isServer then
		--g_currentMission:addMoney(-self:getServiceCost(), self:getOwnerFarmId(), MoneyType.VEHICLE_REPAIR, true, true)
		--self:setDamageAmount(0)
		self.MajorService.lastServiceAt = self.operatingTime
		-- MajorService:setLastServiceText(self.MajorService.lastServiceAt)
		-- MajorService:setNextServiceText(self.MajorService.lastServiceAt, self.operatingTime, self.lifetime)
		self:raiseDirtyFlags(self.spec_wearable.dirtyFlag)
		g_workshopScreen.list:reloadData()
	--end
end


function MajorService:onListSelectionChanged(list, section, index)
	local selectedVehicle = g_workshopScreen.vehicles[index]
	if MajorService.debug then
		print("---- MajorService: onListSelectionChanged ".. selectedVehicle:getName())
	end

	local lastServiceAt = selectedVehicle.MajorService.lastServiceAt
	local operatingTime = selectedVehicle.operatingTime
	local lifetime = selectedVehicle.lifetime	
	local serviceDuration = lifetime * EconomyManager.LIFETIME_OPERATINGTIME_RATIO * 3600000
	local halfDuration = serviceDuration / 2

	local timeSinceLastService = operatingTime - lastServiceAt

	if timeSinceLastService > halfDuration then
		g_workshopScreen.serviceButton.disabled = false
	else
		g_workshopScreen.serviceButton.disabled = true
	end


	g_workshopScreen.serviceButton:setText(string.format(g_i18n:getText("ui_major_service_button"), g_i18n:formatMoney(selectedVehicle:getServiceCost(), 0, true, true)))
	g_workshopScreen.serviceButton.parent:invalidateLayout()
end

function MajorService:setLastServiceText(lastServiceAt)
	local minutes = lastServiceAt / 60000
	local hours = math.floor(minutes / 60)
	minutes = math.floor((minutes - hours * 60) / 6) * 10
	return string.format(g_i18n:getText("shop_operatingTime"), hours, minutes)	
end

function MajorService:setNextServiceText(lastServiceAt, operatingTime, lifetime)

	local serviceDuration = lifetime * EconomyManager.LIFETIME_OPERATINGTIME_RATIO * 3600000
	local halfDuration = serviceDuration / 2

	local timeSinceLastService = operatingTime - lastServiceAt

	if MajorService.debug then
		print("---- MajorService: lastServiceAt - " .. tostring(lastServiceAt))
		print("---- MajorService: operatingTime - " .. tostring(operatingTime))
		print("---- MajorService: lifetime - " .. tostring(lifetime))
		print("---- MajorService: timeSinceLastService - " .. tostring(timeSinceLastService))
		print("---- MajorService: serviceDuration - " .. tostring(serviceDuration))
	end

	if timeSinceLastService >= serviceDuration then
		return "now"
	else
		--local minutes = (serviceDuration - timeSinceLastService) / 60000
		local minutes = (serviceDuration + lastServiceAt) / 60000
		local hours = math.floor(minutes / 60)
		minutes = math.floor((minutes - hours * 60) / 6) * 10
		return string.format(g_i18n:getText("shop_operatingTime"), hours, minutes)
	end	
end

function MajorService.registerFunctions(vehicleType)
	if MajorService.debug then
		print("---- MajorService: registerFunctions")
	end
    SpecializationUtil.registerFunction(vehicleType, "getServiceCost", MajorService.getServiceCost)
    SpecializationUtil.registerFunction(vehicleType, "serviceVehicle", MajorService.serviceVehicle)
end

function MajorService.registerEventListeners(vehicleType)
	if MajorService.debug then
		print("---- MajorService: registerEventListeners")
	end
	local functionNames = {
		"onLoad"
		,"onLoadFinished"
		,"saveToXMLFile"
		,"onReadStream"
		,"onWriteStream"
	}
	
	for i, v in ipairs(functionNames) do
		SpecializationUtil.registerEventListener(vehicleType, v, MajorService);
	end
end

function MajorService:onLoad(savegame)	
	if MajorService.debug then
		print("---- MajorService: onLoad")
	end

	self.MajorService = {};
	self.spec_MajorService = self.MajorService;

	self.MajorService.lastServiceAt = 0;
	self.MajorService.DamageMultiplier = 1;
	self.MajorService.PriceMultiplier = 1;

	if FS25_Configure_Maintenance ~= nil and FS25_Configure_Maintenance.g_r_maintenance ~= nil then
		self.MajorService.DamageMultiplier = FS25_Configure_Maintenance.g_r_maintenance.maintenanceDuration
		self.MajorService.PriceMultiplier = FS25_Configure_Maintenance.g_r_maintenance.maintenanceCost
	end

end

function MajorService:onLoadFinished(savegame)
	if MajorService.debug then
		print("---- MajorService: onLoadFinished ")
	end

	if savegame ~= nil then
		local lastServiceAt = savegame.xmlFile:getValue(savegame.key..".FS25_MajorService.MajorService#lastServiceAt")
		if lastServiceAt ~= nil then
			self.MajorService.lastServiceAt = lastServiceAt * 1000;
			if MajorService.debug then
				print("---- MajorService: got lastServiceAt"..tostring(lastServiceAt*1000))
			end
		end
	end
end

function MajorService:onDelete()
	if MajorService.debug then
		print("---- MajorService: onDelete")
	end
end

function MajorService:saveToXMLFile(xmlFile, key)
	if MajorService.debug then
		print("---- MajorService: saveToXMLFile")
	end
	xmlFile:setValue(key.."#lastServiceAt", self.MajorService.lastServiceAt / 1000)
	xmlFile:setValue(key.."#currentHours", self.operatingTime / 1000)
	--xmlFile:setValue(key .. "#operatingTime", self.operatingTime / 1000)
end

function MajorService:onReadStream(streamId, connection)
	if MajorService.debug then
		print("---- MajorService: onReadStream")
	end
    if connection:getIsServer() then
        self.MajorService.lastServiceAt = streamReadFloat32(streamId) * 1000;
    end
end

function MajorService:onWriteStream(streamId, connection)
	if MajorService.debug then
		print("---- MajorService: onWriteStream")
	end
    if not connection:getIsServer() then
        streamWriteFloat32(streamId, self.MajorService.lastServiceAt / 1000);
    end
end

function MajorService:setLastServiceAt(operatingTime)
	if MajorService.debug then
		print("---- MajorService: setLastServiceAt ".. tostring(operatingTime))
	end
	self.MajorService.lastServiceAt = operatingTime
end

function MajorService:getServiceCost()
	local priceMultiplier = 1
	if FS25_Configure_Maintenance ~= nil and FS25_Configure_Maintenance.g_r_maintenance ~= nil then
		priceMultiplier = FS25_Configure_Maintenance.g_r_maintenance.maintenanceCost
	end

	if MajorService.debug then
		print("---- MajorService: getServiceCost. Price Multiplier - " .. tostring(priceMultiplier))
	end
	local servicePrice = self.price * MajorService.Cost * priceMultiplier
	return servicePrice
end

function MajorService:updateDamageAmount(superFunc, dt)
	local spec = self.spec_wearable

	if FS25_Configure_Maintenance ~= nil and FS25_Configure_Maintenance.g_r_maintenance ~= nil then
		self.MajorService.DamageMultiplier = FS25_Configure_Maintenance.g_r_maintenance.maintenanceDuration
	end

	if MajorService.debug then
		print("---- MajorService: updateDamageAmount -- " .. tostring(self.MajorService.DamageMultiplier))
	end

	if spec:getUsageCausesDamage() then
		local factor = 1

		if spec.lifetime ~= nil and spec.lifetime ~= 0 then
			local ageMultiplier = 0.15 * math.min(spec.age / spec.lifetime, 1)
			local operatingTime = spec.operatingTime / 3600000
			local lastServiceAt = spec.MajorService.lastServiceAt / 3600000
			local operatingTimeMultiplier = 0.85 * math.min((operatingTime - lastServiceAt)/ (spec.lifetime * EconomyManager.LIFETIME_OPERATINGTIME_RATIO), 1)
			-- if MajorService.debug then
			-- 	print("---- MajorService: updateDamageAmount lastServiceAt - ".. tostring(lastServiceAt) .. " operatingTime - " .. tostring(operatingTime))
			-- end
			factor = 1 + EconomyManager.MAX_DAILYUPKEEP_MULTIPLIER * (ageMultiplier + operatingTimeMultiplier)
		end

		return dt * spec.wearDuration * 0.35 * factor * spec.MajorService.DamageMultiplier
	else
		return 0
	end
end

function MajorService.LogMessage(message)
	if self.debug then
		print("---- MajorService: "..message)
	end
end