-- UpgradableFactories: Add upgrade levels to production points

UpgradableFactories = {}

-- Config
UpgradableFactories.MAX_LEVEL = 10
UpgradableFactories.STEP_MULTIPLIER = 0.25 -- +25% throughput per level
UpgradableFactories.COST_FACTOR = 0.20     -- 20% of placeable price per level step

local UF = UpgradableFactories
local UF_MODNAME = g_currentModName or "FS25_UpgradableFactories"

local function dbg(fmt, ...)
    -- Uncomment for verbose logging
    -- print(string.format("[UF] " .. fmt, ...))
end

function UF.getState(pp)
    pp.fs25UF = pp.fs25UF or { level = 0 }
    return pp.fs25UF
end

function UF.getMultiplier(pp)
    local st = UF.getState(pp)
    return 1.0 + (st.level or 0) * UF.STEP_MULTIPLIER
end

function UF.getNextLevelCost(pp)
    local placeable = pp.owningPlaceable
    local basePrice = 100000
    if placeable ~= nil and placeable.getPrice ~= nil then
        basePrice = placeable:getPrice() or basePrice
    end
    local nextLevel = math.min(UF.getState(pp).level + 1, UF.MAX_LEVEL)
    local cost = math.floor((basePrice or 0) * UF.COST_FACTOR * nextLevel)
    return cost
end

function UF.captureBaseValues(pp)
    if pp.productions == nil then return end
    for _, prod in ipairs(pp.productions) do
        if prod.baseCyclesPerMinute == nil then
            prod.baseCyclesPerMinute = prod.cyclesPerMinute or 1
        end
        if prod.baseCyclesPerHour == nil then
            prod.baseCyclesPerHour = prod.cyclesPerHour or ((prod.baseCyclesPerMinute or 1) * 60)
        end
        if prod.baseCyclesPerMonth == nil then
            prod.baseCyclesPerMonth = prod.cyclesPerMonth or (prod.baseCyclesPerHour * 24 * 30)
        end
        if prod.baseCostsPerActiveMinute == nil then
            prod.baseCostsPerActiveMinute = prod.costsPerActiveMinute or 0
        end
        if prod.baseCostsPerActiveHour == nil then
            prod.baseCostsPerActiveHour = prod.costsPerActiveHour or prod.baseCostsPerActiveMinute * 60
        end
        if prod.baseCostsPerActiveMonth == nil then
            prod.baseCostsPerActiveMonth = prod.costsPerActiveMonth or prod.baseCostsPerActiveHour * 24 * 30
        end
    end
end

function UF.applyUpgrade(pp)
    UF.captureBaseValues(pp)
    local mult = UF.getMultiplier(pp)
    for _, prod in ipairs(pp.productions or {}) do
        -- Scale throughput
        if prod.baseCyclesPerMinute ~= nil then prod.cyclesPerMinute = prod.baseCyclesPerMinute * mult end
        if prod.baseCyclesPerHour ~= nil then prod.cyclesPerHour = prod.baseCyclesPerHour * mult end
        if prod.baseCyclesPerMonth ~= nil then prod.cyclesPerMonth = prod.baseCyclesPerMonth * mult end
        -- Scale running costs proportionally to throughput
        if prod.baseCostsPerActiveMinute ~= nil then prod.costsPerActiveMinute = prod.baseCostsPerActiveMinute * mult end
        if prod.baseCostsPerActiveHour ~= nil then prod.costsPerActiveHour = prod.baseCostsPerActiveHour * mult end
        if prod.baseCostsPerActiveMonth ~= nil then prod.costsPerActiveMonth = prod.baseCostsPerActiveMonth * mult end
    end
    if pp.updateFxState ~= nil then
        pp:updateFxState()
    end
end

function UF.setLevel(pp, level)
    local st = UF.getState(pp)
    st.level = math.clamp(level or 0, 0, UF.MAX_LEVEL)
    UF.applyUpgrade(pp)
    if pp.raiseDirtyFlags ~= nil and pp.dirtyFlag ~= nil then
        pp:raiseDirtyFlags(pp.dirtyFlag)
    end
end

-- Write/read custom save data within ProductionPoint save
local orig_PP_save = ProductionPoint.saveToXMLFile
ProductionPoint.saveToXMLFile = function(self, xmlFile, key, usedModNames)
    if orig_PP_save ~= nil then orig_PP_save(self, xmlFile, key, usedModNames) end
    local st = UF.getState(self)
    xmlFile:setInt(string.format("%s.fs25UpgradableFactories#level", key), st.level or 0)
end

local orig_PP_load = ProductionPoint.load
ProductionPoint.load = function(self, components, xmlFile, key, customEnv, i3dMappings)
    local ok = true
    if orig_PP_load ~= nil then
        ok = orig_PP_load(self, components, xmlFile, key, customEnv, i3dMappings)
    end
    if ok then
        UF.captureBaseValues(self)
        UF.applyUpgrade(self)
    end
    return ok
end

local orig_PP_loadSave = ProductionPoint.loadFromXMLFile
ProductionPoint.loadFromXMLFile = function(self, xmlFile, key)
    local ok = true
    if orig_PP_loadSave ~= nil then
        ok = orig_PP_loadSave(self, xmlFile, key)
    end
    local level = xmlFile:getInt(string.format("%s.fs25UpgradableFactories#level", key), 0)
    UF.setLevel(self, level or 0)
    return ok
end

local orig_PP_writeStream = ProductionPoint.writeStream
ProductionPoint.writeStream = function(self, streamId, connection)
    if orig_PP_writeStream ~= nil then
        orig_PP_writeStream(self, streamId, connection)
    end
    if not connection:getIsServer() then
        streamWriteUIntN(streamId, UF.getState(self).level or 0, 4)
    end
end

local orig_PP_readStream = ProductionPoint.readStream
ProductionPoint.readStream = function(self, streamId, connection)
    if orig_PP_readStream ~= nil then
        orig_PP_readStream(self, streamId, connection)
    end
    if connection:getIsServer() then
        local level = streamReadUIntN(streamId, 4) or 0
        UF.setLevel(self, level)
    end
end

-- Inject UI button into production frame
local orig_Menu_updateButtons = InGameMenuProductionFrame.updateMenuButtons
function InGameMenuProductionFrame:updateMenuButtons()
    if orig_Menu_updateButtons ~= nil then orig_Menu_updateButtons(self) end

    -- Only when owned and a production point is selected
    local prod, pp = self:getSelectedProduction()
    if self.pointsSelector:getState() ~= InGameMenuProductionFrame.POINTS_OWNED then return end
    if pp == nil or pp.owningPlaceable == nil then return end

    -- Build/refresh upgrade button
    self.upgradeButtonInfo = self.upgradeButtonInfo or {}
    self.upgradeButtonInfo.inputAction = InputAction.MENU_EXTRA_1

    local level = (pp.fs25UF and pp.fs25UF.level) or 0
    if level >= UF.MAX_LEVEL then
        local txt = g_i18n:hasText("ui_upgradeFactoryMax", UF_MODNAME) and g_i18n:getText("ui_upgradeFactoryMax", UF_MODNAME) or "Max Level Reached"
        self.upgradeButtonInfo.text = txt
        self.upgradeButtonInfo.callback = function() end
        self.upgradeButtonInfo.profile = "buttonDisabled"
    else
        local stepPct = math.floor(UF.STEP_MULTIPLIER * 100)
        local cost = UF.getNextLevelCost(pp)
        local moneyStr = g_i18n:formatMoney(cost, 0, true, true)
        local tmpl = g_i18n:hasText("ui_upgradeFactoryBtn", UF_MODNAME) and g_i18n:getText("ui_upgradeFactoryBtn", UF_MODNAME) or "Upgrade (+%d%%) %s"
        self.upgradeButtonInfo.text = string.format(tmpl, stepPct, moneyStr)
        self.upgradeButtonInfo.callback = function()
            UpgradableFactories.onUpgradePressed(self)
        end
        self.upgradeButtonInfo.profile = "buttonOK"
    end

    table.insert(self.menuButtonInfo, self.upgradeButtonInfo)
    self:setMenuButtonInfoDirty()
end

-- Augment details panel with current/next tier info
local orig_populateCell = InGameMenuProductionFrame.populateCellForItemInSection
function InGameMenuProductionFrame:populateCellForItemInSection(list, section, index, cell)
    if orig_populateCell ~= nil then
        orig_populateCell(self, list, section, index, cell)
    end
    -- Only add info in owned details list on the detailsCell
    if list ~= self.detailsList then return end
    if self.pointsSelector:getState() ~= InGameMenuProductionFrame.POINTS_OWNED then return end
    if cell == nil or cell.name ~= "detailsCell" then return end

    local prod, pp = self:getSelectedProduction()
    if prod == nil or pp == nil then return end

    local level = (pp.fs25UF and pp.fs25UF.level) or 0
    local currMult = 1.0 + level * UF.STEP_MULTIPLIER
    local nextLevel = math.min(level + 1, UF.MAX_LEVEL)
    local nextMult = 1.0 + nextLevel * UF.STEP_MULTIPLIER
    -- Add dedicated rows: Current tier, Next tier, Upgrade price
    local infoCycles = cell:getAttribute("infoCycles") or (cell.getDescendantByName and cell:getDescendantByName("infoCycles")) or nil
    local infoCosts = cell:getAttribute("infoCosts") or (cell.getDescendantByName and cell:getDescendantByName("infoCosts")) or nil

    if infoCycles ~= nil and infoCosts ~= nil then
        local function getLocText(key, fallback)
            if key ~= nil and g_i18n ~= nil and g_i18n.hasText ~= nil and g_i18n:hasText(key, UF_MODNAME) then
                return g_i18n:getText(key, UF_MODNAME)
            end
            return fallback
        end
        -- Create once
        if not cell.fs25UFInstalled then
            local infoStatus = cell:getAttribute("infoStatus")
            local infoBox = infoStatus ~= nil and infoStatus.parent or nil
            if infoBox ~= nil then
                local infoElements = infoBox.elements or {}
                local infoCostsIdx
                for i, el in ipairs(infoElements) do
                    if el == infoCosts then
                        infoCostsIdx = i
                        break
                    end
                end
                local labelTemplate = (infoCostsIdx ~= nil and infoElements[infoCostsIdx - 1]) or nil
                local bgTemplate = (infoCostsIdx ~= nil and infoElements[infoCostsIdx - 2]) or infoElements[1]
                local rowStep = 0
                if infoCycles.position ~= nil and infoCosts.position ~= nil then
                    rowStep = (infoCosts.position[2] or 0) - (infoCycles.position[2] or 0)
                end
                if rowStep == 0 then
                    rowStep = -0.04
                elseif rowStep > 0 then
                    rowStep = -rowStep
                end

                cell.fs25UFOriginalCellHeight = cell.fs25UFOriginalCellHeight or ((cell.size and cell.size[2]) or 0)
                local baseRowCount = 0
                if infoStatus ~= nil then baseRowCount = baseRowCount + 1 end
                if infoCycles ~= nil then baseRowCount = baseRowCount + 1 end
                if infoCosts ~= nil then baseRowCount = baseRowCount + 1 end

                cell.fs25UFRowData = {
                    rowStep = rowStep,
                    baseLabelY = (labelTemplate ~= nil and labelTemplate.position ~= nil and labelTemplate.position[2]) or 0,
                    baseValueY = (infoCosts.position and infoCosts.position[2]) or 0,
                    baseBgY = (bgTemplate ~= nil and bgTemplate.position ~= nil and bgTemplate.position[2]) or 0,
                    baseRowCount = baseRowCount,
                }
                cell.fs25UFTierRows = {}

                local function createRow(labelName, valueName, labelKey, fallbackText)
                    local idx = #cell.fs25UFTierRows + 1
                    local offset = rowStep * idx
                    local rowNumber = (cell.fs25UFRowData.baseRowCount or 3) + idx

                    if bgTemplate ~= nil and bgTemplate.clone ~= nil and rowNumber % 2 == 1 then
                        local bgClone = bgTemplate:clone(infoBox)
                        bgClone.name = valueName .. "Bg"
                        if bgClone.setPosition ~= nil then
                            bgClone:setPosition(nil, cell.fs25UFRowData.baseBgY + offset)
                        end
                        infoBox:addElement(bgClone)
                    end

                    if labelTemplate ~= nil and labelTemplate.clone ~= nil then
                        local labelClone = labelTemplate:clone(infoBox)
                        labelClone.name = labelName
                        if labelClone.setPosition ~= nil then
                            labelClone:setPosition(nil, cell.fs25UFRowData.baseLabelY + offset)
                        end
                        if labelClone.setText ~= nil then
                            labelClone:setText(getLocText(labelKey, fallbackText))
                        end
                        infoBox:addElement(labelClone)
                    end

                    local valueClone = infoCosts:clone(infoBox)
                    valueClone.name = valueName
                    if valueClone.setPosition ~= nil then
                        valueClone:setPosition(nil, cell.fs25UFRowData.baseValueY + offset)
                    end
                    infoBox:addElement(valueClone)

                    table.insert(cell.fs25UFTierRows, {label = labelName, value = valueName})
                end

                createRow("titleTierCurrent", "infoTierCurrent", "ui_upgradeFactoryCurrent", "Current tier")
                createRow("titleTierNext", "infoTierNext", "ui_upgradeFactoryNext", "Next tier")
                createRow("titleTierPrice", "infoTierPrice", "ui_upgradeFactoryCost", "Upgrade price")

                local addedHeight = math.abs(rowStep) * #cell.fs25UFTierRows
                if infoBox.size ~= nil then
                    infoBox:setSize(nil, (infoBox.size[2] or 0) + addedHeight)
                end
                if cell.size ~= nil then
                    cell:setSize(nil, (cell.size[2] or 0) + addedHeight)
                end
                if self.detailsList ~= nil and self.detailsList.cellDatabase ~= nil and self.detailsList.cellDatabase.detailsCell ~= nil and self.detailsList.cellDatabase.detailsCell.size ~= nil then
                    self.detailsList.cellDatabase.detailsCell.size[2] = (cell.size and cell.size[2]) or self.detailsList.cellDatabase.detailsCell.size[2]
                end
                if infoBox.invalidateLayout ~= nil then
                    infoBox:invalidateLayout()
                end
                if cell.invalidateLayout ~= nil then
                    cell:invalidateLayout()
                end
            end

            cell.fs25UFInstalled = true
        end

        -- Update texts each refresh
        local function setPair(leftName, rightName, leftKey, leftFallback, rightText)
            local left = (cell.getDescendantByName and cell:getDescendantByName(leftName)) or nil
            local right = (cell.getDescendantByName and cell:getDescendantByName(rightName)) or nil
            if left ~= nil and left.setText ~= nil then
                left:setText(getLocText(leftKey, leftFallback))
            end
            if right ~= nil then
                if right.setValue ~= nil then
                    right:setValue(rightText)
                elseif right.setText ~= nil then
                    right:setText(rightText)
                end
            end
        end
        setPair("titleTierCurrent", "infoTierCurrent", "ui_upgradeFactoryCurrent", "Current tier", string.format("%d (x%.2f)", level, currMult))
        setPair("titleTierNext", "infoTierNext", "ui_upgradeFactoryNext", "Next tier", string.format("%d (x%.2f)", nextLevel, nextMult))
        local priceText = level < UF.MAX_LEVEL and g_i18n:formatMoney(UF.getNextLevelCost(pp), 0, true, true) or "-"
        setPair("titleTierPrice", "infoTierPrice", "ui_upgradeFactoryCost", "Upgrade price", priceText)
    end
end

function UF.onUpgradePressed(frame)
    local _, pp = frame:getSelectedProduction()
    if pp == nil then return end
    local st = UF.getState(pp)
    if st.level >= UF.MAX_LEVEL then return end

    local farmId = (pp.getOwnerFarmId ~= nil) and pp:getOwnerFarmId() or g_currentMission:getFarmId()
    local cost = UF.getNextLevelCost(pp)
    local nextLevel = st.level + 1
    local nextMult = 1.0 + nextLevel * UF.STEP_MULTIPLIER

    local title = g_i18n:hasText("ui_upgradeFactoryTitle", UF_MODNAME) and g_i18n:getText("ui_upgradeFactoryTitle", UF_MODNAME) or "Upgrade Factory"
    local detailsTmpl = g_i18n:hasText("ui_upgradeFactoryDialog", UF_MODNAME) and g_i18n:getText("ui_upgradeFactoryDialog", UF_MODNAME) or "Level %d -> %d\nThroughput x%.2f\nCost %s\nProceed?"
    local text = string.format(detailsTmpl, st.level, nextLevel, nextMult, g_i18n:formatMoney(cost, 0, true, true))

    local function doUpgrade()
        -- Debounce: only upgrade once per dialog
        if frame.fs25UFConfirmOpen then
            frame.fs25UFConfirmOpen = false
        end
        UpgradeFactoryEvent.sendEvent(pp, nextLevel, cost, farmId)
        if frame.updateProductionLists ~= nil then
            frame:updateProductionLists()
        end
    end

    -- Open confirmation dialog with explicit OK/Cancel texts
    local okText = g_i18n:getText("button_ok")
    local cancelText = g_i18n:getText("button_cancel")
    frame.fs25UFConfirmOpen = true
    if YesNoDialog ~= nil and YesNoDialog.show ~= nil then
        YesNoDialog.show(function(choice)
            local yes = choice == true
            if yes then
                doUpgrade()
            else
                frame.fs25UFConfirmOpen = false
            end
        end, nil, text, title, okText, cancelText)
    else
        -- Fallback: require MENU_ACCEPT to proceed when no dialog system exists
        frame.fs25UFConfirmOpen = false
        doUpgrade()
    end
end

dbg("UpgradableFactories loaded and hooks installed.")

-- Safety patch: Guard greenhouse distribution on empty active filltypes to avoid nil/index errors
-- Note: Do not touch PlaceableGreenhouse. Any greenhouse errors should be unrelated to this mod now.
